/*
 * Copyright 2007 Robert Hanson <iamroberthanson AT gmail.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gwtwidgets.client.ui.canvas;

import java.util.List;

import org.gwtwidgets.client.ui.canvas.impl.CanvasFactory;
import org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl;
import org.gwtwidgets.client.ui.canvas.impl.IECanvasImpl;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.ClickListenerCollection;
import com.google.gwt.user.client.ui.FocusListener;
import com.google.gwt.user.client.ui.FocusListenerCollection;
import com.google.gwt.user.client.ui.KeyboardListener;
import com.google.gwt.user.client.ui.KeyboardListenerCollection;
import com.google.gwt.user.client.ui.MouseListener;
import com.google.gwt.user.client.ui.MouseListenerCollection;
import com.google.gwt.user.client.ui.MouseWheelListener;
import com.google.gwt.user.client.ui.MouseWheelListenerCollection;
import com.google.gwt.user.client.ui.SourcesClickEvents;
import com.google.gwt.user.client.ui.SourcesFocusEvents;
import com.google.gwt.user.client.ui.SourcesKeyboardEvents;
import com.google.gwt.user.client.ui.SourcesMouseEvents;
import com.google.gwt.user.client.ui.SourcesMouseWheelEvents;
import com.google.gwt.user.client.ui.Widget;

/**
 * <p>
 * Abstract base class for canvas widgets. A canvas is a rectangular area which
 * can display vector graphics - scalable shapes such as lines, polygons and
 * images. There exist currently two implementations: the {@link FFCanvasImpl}
 * which runs on browsers which support the <code>canvas</code> widget and the
 * {@link IECanvasImpl} widget which emulates a canvas in Internet Explorer
 * through VML.
 * </p>
 * <p>
 * Canvas implementations have no public constructor, instead they are derived
 * from a {@link CanvasFactory}. The GWT compiler makes sure that
 * {@link CanvasFactory} returns the correct canvas implementation for each
 * browser.
 * </p>
 * <p>
 * Canvas instances are stateful, similar to a java <code>Graphics</code>
 * object. A stroke, filling, rotation etc. applied affects all subsequent
 * operations. Similar to a stream, a canvas must be flushed for the graphics
 * operations to become visible - the opposite does not hold; there is no
 * guarantee that graphics operations will not be visible without flushing.
 * </p>
 * <p>
 * The typical lifecycle of a canvas widget looks like:<br>
 * 
 * <pre>
 * Canvas canvas = Canvas.create(640,480);
 * panel.add(canvas);
 * ...
 * canvas.clear();
 * ...
 * canvas.setStroke(255,0,0,1);
 * canvas.drawLine(0,0,100,100);
 * ...
 * canvas.flush();
 * </pre>
 * </p>
 * <p>
 * Note that in order for the canvas to function in Internet Explorer, a special
 * namespace must be used on the HTML page and a special CSS style must be
 * included. For details please consult {@link IECanvasImpl}.
 * </p>
 * <p>
 * Some implementations may not allow you to draw shapes starting at greater coordinates 
 * towards lower coordinates, i.e. <code>drawRectangle(100,100,10,10)</code> may fail on 
 * certain browsers.
 * </p>
 * 
 * <em>TODO:</em> Define the exact semantics of rotation as they currently vary across implementations. 
 * <em>TODO:</em> Formalise whether borders extend or overlay shapes. 
 * 
 * @author George Georgovassilis g.georgovassilis[at]gmail.com
 * @see IECanvasImpl, FFCanvasImpl
 */
public abstract class Canvas extends Widget implements SourcesClickEvents, SourcesFocusEvents, SourcesKeyboardEvents,
		SourcesMouseEvents, SourcesMouseWheelEvents {

	private ClickListenerCollection clickListeners;
	private FocusListenerCollection focusListeners;
	private KeyboardListenerCollection keyboardListeners;
	private MouseListenerCollection mouseListeners;
	private MouseWheelListenerCollection mouseWheelListeners;
	
	/**
	 * Rotation in radians.
	 */
	protected double rotation = 0;

	private void addListener(int eventBits, Object listener, List collection) {
		if (collection.size() == 0)
			DOM.sinkEvents(getElement(), DOM.getEventsSunk(getElement()) | eventBits);
		collection.add(listener);
	}

	private void removeListener(int eventBits, Object listener, List collection) {
		if (collection == null) return;
		collection.remove(listener);
		if (collection.size() == 0)
			DOM.sinkEvents(getElement(), (DOM.getEventsSunk(getElement()) | eventBits) ^ eventBits);
	}

	public void onBrowserEvent(Event event) {
		switch (DOM.eventGetType(event)) {
		case Event.ONCLICK:
			if (clickListeners != null)
				clickListeners.fireClick(this);
			break;
		case Event.ONFOCUS:
		case Event.ONBLUR:
			if (focusListeners != null)
				focusListeners.fireFocusEvent(this, event);
			break;
		case Event.ONKEYDOWN:
		case Event.ONKEYUP:
		case Event.ONKEYPRESS:
			if (keyboardListeners != null)
				keyboardListeners.fireKeyboardEvent(this, event);
			break;
		case Event.ONMOUSEDOWN:
		case Event.ONMOUSEMOVE:
		case Event.ONMOUSEOUT:
		case Event.ONMOUSEOVER:
		case Event.ONMOUSEUP:
			if (mouseListeners != null)
				mouseListeners.fireMouseEvent(this, event);
			break;
		case Event.ONMOUSEWHEEL:
			if (mouseWheelListeners != null)
				mouseWheelListeners.fireMouseWheelEvent(this, event);
			break;
		}
	}

	/**
	 * Applies a rotation to all following drawing operations.
	 * 
	 * @param angle
	 *            Angle in radians
	 */
	public void setRotation(double angle){
		this.rotation = angle;
	}

	/**
	 * Gets the currently set rotation
	 * 
	 * @return angle in radians
	 */
	public double getRotation(){
		return rotation;
	}

	/**
	 * Applies a stroke (outline) to all drawings. For lines and polylines this
	 * is the appearance of the line.
	 * 
	 * @param red
	 *            Red component from 0 ( =none ) to 255 (full),
	 * @param green
	 *            Green component from 0 ( =none ) to 255 (full),
	 * @param blue
	 *            Blue component from 0 ( =none ) to 255 (full),
	 * @param alpha
	 *            Alpha (opacity) component from 0 (=transparent) to 1 (=opaque)
	 */
	public abstract void setStroke(int red, int green, int blue, double alpha);

	/**
	 * Applies a filling (interior) to surfaces. This affects
	 * {@link #drawImage(Element, double, double, double, double, double, double)},
	 * {@link #drawPolygon(double[], double[])}, and
	 * {@link #drawText(String, double, double)}.
	 * 
	 * @param red
	 *            Red component from 0 ( =none ) to 255 (full),
	 * @param green
	 *            Green component from 0 ( =none ) to 255 (full),
	 * @param blue
	 *            Blue component from 0 ( =none ) to 255 (full),
	 * @param alpha
	 *            Alpha (opacity) component from 0 (=transparent) to 1 (=opaque)
	 */
	public abstract void setFill(int red, int green, int blue, double alpha);

	/**
	 * Applies stroke width to strokes (outlines). Defaults to 1.
	 * 
	 * @param weight
	 */
	public abstract void setStrokeWeight(double weight);

	/**
	 * Apply offset to all following drawings.
	 * 
	 * @param left
	 *            offset to be added to the X ordinate
	 * @param top
	 *            offset to be added to the Y ordinate
	 */
	public abstract void setOffset(double left, double top);

	/**
	 * Draws a rectangle with the stroke as border and fills it with the current
	 * fill.
	 * 
	 * @param left
	 * @param top
	 * @param width
	 * @param height
	 */

	public abstract void drawRectangle(double left, double top, double width, double height);

	/**
	 * Draws a line.
	 * 
	 * @param fromLeft
	 * @param fromTop
	 * @param toLeft
	 * @param toTop
	 */
	public abstract void drawLine(double fromLeft, double fromTop, double toLeft, double toTop);

	/**
	 * Draws an arc with the stroke as border and fills it with the current
	 * fill.
	 * 
	 * @param centerLeft
	 * @param centerTop
	 * @param width
	 * @param height
	 * @param fromAngle
	 *            Angle in radians.
	 * @param toAngle
	 *            Angle in radians.
	 */
	public abstract void drawArc(double centerLeft, double centerTop, double width, double height, double fromAngle, double toAngle);

	/**
	 * Draws a consecutive line with the current stroke as border.
	 * 
	 * @param x
	 *            X ordinates of each line node
	 * @param y
	 *            Y ordinates of each line node.
	 */
	public abstract void drawPolyLine(double x[], double y[]);

	/**
	 * Draws a closed polygon with the current stroke as border and fills it
	 * with the current fill. It is not defined how
	 * <a href="http://en.wikipedia.org/wiki/Complex_polygon">convex polygons<a>
	 * are to be filled.
	 * 
	 * @param x
	 *            X ordinates of each line node
	 * @param y
	 *            Y ordinates of each line node.
	 */
	public abstract void drawPolygon(double x[], double y[]);

	/**
	 * Clears the canvas area and resets fill, stroke, strokeWeight, font and
	 * offset.
	 */
	public abstract void clear();

	/**
	 * Makes all changes performed since the last flush visible.
	 */
	public abstract void flush();

	/**
	 * Draws an image.
	 * 
	 * @param image
	 *            Image element. Please note that this must be a genuine
	 *            <code>&lt;img&gt;</code> tag element, using a clipped GWT
	 *            image will not work. Images are rotated around their center and not
	 *            around the center of the coordinate system.
	 * @param sx
	 *            Offset from the left in the image from which to start drawing
	 * @param sy
	 *            Offset from the top in the image from which to start drawing
	 * @param swidth
	 *            Width of the part in the image which to draw
	 * @param sheight
	 *            Height of the part in the image which to draw
	 * @param dx
	 *            X position in the canvas to draw the selected image area to
	 * @param dy
	 *            Y position in the canvas to draw the selected image area to
	 * @param dwidth
	 *            Width of the destination area. Can be used for scaling.
	 * @param dheight
	 *            Height of the destination area.
	 */
	public abstract void drawImage(Element image, double sx, double sy, double swidth, double sheight, double dx,
			double dy, double dwidth, double dheight);

	/**
	 * Draw Text. {@link #setFont(BitmapFontImpl)} must be called prior to
	 * drawing the first text.
	 * 
	 * @param text
	 * @param x
	 *            X ordinate of the canvas location to draw text to.
	 * @param y
	 *            Y ordinate of the canvas location to draw text to.
	 */
	public abstract void drawText(String text, double x, double y);

	/**
	 * Apply font to all following text drawings.
	 * 
	 * @param font
	 */
	public abstract void setFont(Font font);

	private static CanvasFactory factory = null;

	private static CanvasFactory getFactoryImpl() {
		if (factory != null)
			return factory;
		factory = (CanvasFactory) GWT.create(CanvasFactory.class);
		return factory;
	}

	/**
	 * Return a suitable canvas for the browser the application is running in.
	 * 
	 * @param width
	 *            Width in pixels
	 * @param height
	 *            Height in pixels
	 * @return
	 */
	public static Canvas create(int width, int height) {
		return getFactoryImpl().create(width, height);
	}

	/**
	 * Creates a {@link Font} object from a bitmap representation.
	 * 
	 * @param bitmapPath
	 *            URL to bitmap as created by the {@link BitmapFontCreator}
	 * @param descriptionPath
	 *            URL to the font description path as created by the
	 *            {@link BitmapFontCreator}
	 * @param listener
	 *            Optional listener to notify when a font is loaded or an error
	 *            occurred.
	 * @return
	 */
	public static Font createBitmapFont(String bitmapPath, String descriptionPath, FontLoadListener listener) {
		return getFactoryImpl().getFont(bitmapPath, descriptionPath, listener);
	}

	public void addClickListener(ClickListener listener) {
		if (clickListeners == null)
			clickListeners = new ClickListenerCollection();
		addListener(Event.ONCLICK, listener, clickListeners);
	}

	public void removeClickListener(ClickListener listener) {
		removeListener(Event.ONCLICK, listener, clickListeners);
	}

	public void addFocusListener(FocusListener listener) {
		if (focusListeners == null)
			focusListeners = new FocusListenerCollection();
		addListener(Event.ONFOCUS, listener, focusListeners);
	}

	public void removeFocusListener(FocusListener listener) {
		removeListener(Event.ONFOCUS, listener, focusListeners);
	}

	public void addKeyboardListener(KeyboardListener listener) {
		if (keyboardListeners == null)
			keyboardListeners = new KeyboardListenerCollection();
		addListener(Event.KEYEVENTS, listener, keyboardListeners);
	}

	public void removeKeyboardListener(KeyboardListener listener) {
		removeListener(Event.KEYEVENTS, listener, keyboardListeners);
	}

	public void addMouseListener(MouseListener listener) {
		if (mouseListeners == null)
			mouseListeners = new MouseListenerCollection();
		addListener(Event.MOUSEEVENTS, listener, mouseListeners);
	}

	public void removeMouseListener(MouseListener listener) {
		removeListener(Event.MOUSEEVENTS, listener, mouseListeners);
	}

	public void addMouseWheelListener(MouseWheelListener listener) {
		if (mouseWheelListeners == null)
			mouseWheelListeners = new MouseWheelListenerCollection();
		addListener(Event.ONMOUSEWHEEL, listener, mouseListeners);
	}

	public void removeMouseWheelListener(MouseWheelListener listener) {
		removeListener(Event.ONMOUSEWHEEL, listener, mouseWheelListeners);
	}
	
	/**
	 * On some platforms, it is faster to use prepared images rather than the source image
	 * directly. The following code demonstrates how to handle prepared images:<br><br>
	 * <code>
	 * Element image = DOM.createImage();<br>
	 * DOM.setImgSrc(image,"test.png");<br>
	 * Element preparedImage = canvas.prepareImage(image);<br>
	 * canvas.drawImage(preparedImage,0,0,100,100,0,0,100,100);<br>
	 * </code>
	 * 
	 * @param image
	 * @return A prepared image
	 */
	public abstract Element prepareImage(Element image);
	
}
