'use strict';

exports.__esModule = true;
exports.default = tokenizer;
var SINGLE_QUOTE = 39;
var DOUBLE_QUOTE = 34;
var BACKSLASH = 92;
var SLASH = 47;
var NEWLINE = 10;
var SPACE = 32;
var FEED = 12;
var TAB = 9;
var CR = 13;
var OPEN_SQUARE = 91;
var CLOSE_SQUARE = 93;
var OPEN_PARENTHESES = 40;
var CLOSE_PARENTHESES = 41;
var OPEN_CURLY = 123;
var CLOSE_CURLY = 125;
var SEMICOLON = 59;
var ASTERISK = 42;
var COLON = 58;
var AT = 64;

var RE_AT_END = /[ \n\t\r\f\{\(\)'"\\;/\[\]#]/g;
var RE_WORD_END = /[ \n\t\r\f\(\)\{\}:;@!'"\\\]\[#]|\/(?=\*)/g;
var RE_BAD_BRACKET = /.[\\\/\("'\n]/;

function tokenizer(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    var css = input.css.valueOf();
    var ignore = options.ignoreErrors;

    var code = void 0,
        next = void 0,
        quote = void 0,
        lines = void 0,
        last = void 0,
        content = void 0,
        escape = void 0,
        nextLine = void 0,
        nextOffset = void 0,
        escaped = void 0,
        escapePos = void 0,
        prev = void 0,
        n = void 0,
        currentToken = void 0;

    var length = css.length;
    var offset = -1;
    var line = 1;
    var pos = 0;
    var buffer = [];
    var returned = [];

    function unclosed(what) {
        throw input.error('Unclosed ' + what, line, pos - offset);
    }

    function endOfFile() {
        return returned.length === 0 && pos >= length;
    }

    function nextToken() {
        if (returned.length) return returned.pop();
        if (pos >= length) return;

        code = css.charCodeAt(pos);
        if (code === NEWLINE || code === FEED || code === CR && css.charCodeAt(pos + 1) !== NEWLINE) {
            offset = pos;
            line += 1;
        }

        switch (code) {
            case NEWLINE:
            case SPACE:
            case TAB:
            case CR:
            case FEED:
                next = pos;
                do {
                    next += 1;
                    code = css.charCodeAt(next);
                    if (code === NEWLINE) {
                        offset = next;
                        line += 1;
                    }
                } while (code === SPACE || code === NEWLINE || code === TAB || code === CR || code === FEED);

                currentToken = ['space', css.slice(pos, next)];
                pos = next - 1;
                break;

            case OPEN_SQUARE:
                currentToken = ['[', '[', line, pos - offset];
                break;

            case CLOSE_SQUARE:
                currentToken = [']', ']', line, pos - offset];
                break;

            case OPEN_CURLY:
                currentToken = ['{', '{', line, pos - offset];
                break;

            case CLOSE_CURLY:
                currentToken = ['}', '}', line, pos - offset];
                break;

            case COLON:
                currentToken = [':', ':', line, pos - offset];
                break;

            case SEMICOLON:
                currentToken = [';', ';', line, pos - offset];
                break;

            case OPEN_PARENTHESES:
                prev = buffer.length ? buffer.pop()[1] : '';
                n = css.charCodeAt(pos + 1);
                if (prev === 'url' && n !== SINGLE_QUOTE && n !== DOUBLE_QUOTE && n !== SPACE && n !== NEWLINE && n !== TAB && n !== FEED && n !== CR) {
                    next = pos;
                    do {
                        escaped = false;
                        next = css.indexOf(')', next + 1);
                        if (next === -1) {
                            if (ignore) {
                                next = pos;
                                break;
                            } else {
                                unclosed('bracket');
                            }
                        }
                        escapePos = next;
                        while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                            escapePos -= 1;
                            escaped = !escaped;
                        }
                    } while (escaped);

                    currentToken = ['brackets', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

                    pos = next;
                } else {
                    next = css.indexOf(')', pos + 1);
                    content = css.slice(pos, next + 1);

                    if (next === -1 || RE_BAD_BRACKET.test(content)) {
                        currentToken = ['(', '(', line, pos - offset];
                    } else {
                        currentToken = ['brackets', content, line, pos - offset, line, next - offset];
                        pos = next;
                    }
                }

                break;

            case CLOSE_PARENTHESES:
                currentToken = [')', ')', line, pos - offset];
                break;

            case SINGLE_QUOTE:
            case DOUBLE_QUOTE:
                quote = code === SINGLE_QUOTE ? '\'' : '"';
                next = pos;
                do {
                    escaped = false;
                    next = css.indexOf(quote, next + 1);
                    if (next === -1) {
                        if (ignore) {
                            next = pos + 1;
                            break;
                        } else {
                            unclosed('string');
                        }
                    }
                    escapePos = next;
                    while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                        escapePos -= 1;
                        escaped = !escaped;
                    }
                } while (escaped);

                content = css.slice(pos, next + 1);
                lines = content.split('\n');
                last = lines.length - 1;

                if (last > 0) {
                    nextLine = line + last;
                    nextOffset = next - lines[last].length;
                } else {
                    nextLine = line;
                    nextOffset = offset;
                }

                currentToken = ['string', css.slice(pos, next + 1), line, pos - offset, nextLine, next - nextOffset];

                offset = nextOffset;
                line = nextLine;
                pos = next;
                break;

            case AT:
                RE_AT_END.lastIndex = pos + 1;
                RE_AT_END.test(css);
                if (RE_AT_END.lastIndex === 0) {
                    next = css.length - 1;
                } else {
                    next = RE_AT_END.lastIndex - 2;
                }

                currentToken = ['at-word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

                pos = next;
                break;

            case BACKSLASH:
                next = pos;
                escape = true;
                while (css.charCodeAt(next + 1) === BACKSLASH) {
                    next += 1;
                    escape = !escape;
                }
                code = css.charCodeAt(next + 1);
                if (escape && code !== SLASH && code !== SPACE && code !== NEWLINE && code !== TAB && code !== CR && code !== FEED) {
                    next += 1;
                }

                currentToken = ['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

                pos = next;
                break;

            default:
                if (code === SLASH && css.charCodeAt(pos + 1) === ASTERISK) {
                    next = css.indexOf('*/', pos + 2) + 1;
                    if (next === 0) {
                        if (ignore) {
                            next = css.length;
                        } else {
                            unclosed('comment');
                        }
                    }

                    content = css.slice(pos, next + 1);
                    lines = content.split('\n');
                    last = lines.length - 1;

                    if (last > 0) {
                        nextLine = line + last;
                        nextOffset = next - lines[last].length;
                    } else {
                        nextLine = line;
                        nextOffset = offset;
                    }

                    currentToken = ['comment', content, line, pos - offset, nextLine, next - nextOffset];

                    offset = nextOffset;
                    line = nextLine;
                    pos = next;
                } else {
                    RE_WORD_END.lastIndex = pos + 1;
                    RE_WORD_END.test(css);
                    if (RE_WORD_END.lastIndex === 0) {
                        next = css.length - 1;
                    } else {
                        next = RE_WORD_END.lastIndex - 2;
                    }

                    currentToken = ['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

                    buffer.push(currentToken);

                    pos = next;
                }

                break;
        }

        pos++;
        return currentToken;
    }

    function back(token) {
        returned.push(token);
    }

    return {
        back: back,
        nextToken: nextToken,
        endOfFile: endOfFile
    };
}
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
